%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Bootstrap-Based Wald Tests for Structural Parameters %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Get the betas for testing under heteroskedasticity
beta_hat_1_main_column_1 = [B_nG_1(2,1)/B_nG_1(1,1); B_nG_1(3,1)/B_nG_1(1,1)];
beta_hat_2_main_column_1 = [B_nG_2(2,1)/B_nG_2(1,1); B_nG_2(3,1)/B_nG_2(1,1)];

beta_hat_1_main_column_2 = [B_nG_1(2,2)/B_nG_1(1,2); B_nG_1(3,2)/B_nG_1(1,2)];
beta_hat_2_main_column_2 = [B_nG_2(2,2)/B_nG_2(1,2); B_nG_2(3,2)/B_nG_2(1,2)];

beta_hat_1_main_column_3 = [B_nG_1(2,3)/B_nG_1(1,3); B_nG_1(3,3)/B_nG_1(1,3)];
beta_hat_2_main_column_3 = [B_nG_2(2,3)/B_nG_2(1,3); B_nG_2(3,3)/B_nG_2(1,3)];

% Get the estimated thetas which fit sign and permutation of Bs above
[~, ~, Q_hat_1, ~] = rqdecomposition(K, B_nG_1);
[~, ~, Q_hat_2, ~] = rqdecomposition(K, B_nG_2);
theta_hat_1_main = thetas_from_Q_3x3(Q_hat_1);
theta_hat_2_main = thetas_from_Q_3x3(Q_hat_2);

%%%%%%%%%%%%%
% Bootstrap %
%%%%%%%%%%%%%
S = 2000;

denom_row = [1 3 1]; %define which denominator to use in each column when calculating the ratios

theta_hat_boot_1 = zeros(3, S);
theta_hat_boot_2 = zeros(3, S);
B_hat_boot_1     = zeros(3, 3, S);
B_hat_boot_2     = zeros(3, 3, S);

u1_boot = Uhat_1;
u2_boot = Uhat_2;
s       = 1;
rseed   = 1;

while s < S + 1
    
    % Empirically standardized reduced-form shocks
    u_star1 = chol(cov(u1_boot'), 'lower') \ u1_boot;
    u_star2 = chol(cov(u2_boot'), 'lower') \ u2_boot;

    % Initial theta values
    theta01 = theta_hat_1';
    theta02 = theta_hat_2';

    % Optimization setup
    A = []; b = [];
    Aeq = []; beq = [];
    lb = []; ub = [];

    % Estimation
    [theta_hat_1_est, fval1] = fmincon(@(theta) -loglik_t_3x3(u_star1, v, theta), ...
                                       theta01, A, b, Aeq, beq, lb, ub, [], options);
    [theta_hat_2_est, fval2] = fmincon(@(theta) -loglik_t_3x3(u_star2, v, theta), ...
                                       theta02, A, b, Aeq, beq, lb, ub, [], options);

    % Construct B matrices
    B_hat_1 = chol(cov(u1_boot'), 'lower') * Qdim3_prime_loglik(theta_hat_1_est)';
    B_hat_2 = chol(cov(u2_boot'), 'lower') * Qdim3_prime_loglik(theta_hat_2_est)';

    % Take care of permutation and sign
    B_hat_1 = chop(B_hat_1, nod);
    B_hat_2 = chop(B_hat_2, nod);
    B_hat_1 = ColumnSigns_via_correlation(B_hat_1, B_nG_1, K);
    B_hat_2 = ColumnSigns_via_correlation(B_hat_2, B_nG_2, K);

    % Retrieve thetas
    [~, ~, Q_hat_1, ~] = rqdecomposition(K, B_hat_1);
    [~, ~, Q_hat_2, ~] = rqdecomposition(K, B_hat_2);
    theta_hat_1_est = thetas_from_Q_3x3(Q_hat_1);
    theta_hat_2_est = thetas_from_Q_3x3(Q_hat_2);

    % Sign check
    sign_ok = isequal(sign(B_hat_1), sign(B_nG_1)) && ...
              isequal(sign(B_hat_2), sign(B_nG_2));

    if sign_ok
        B_hat_boot_1(:, :, s)   = B_hat_1;
        B_hat_boot_2(:, :, s)   = B_hat_2;
        theta_hat_boot_1(:, s)  = theta_hat_1_est;
        theta_hat_boot_2(:, s)  = theta_hat_2_est;
        s = s + 1;
    end

    % Resample bootstrap data
    stream = RandStream('dsfmt19937', 'Seed', rseed);
    rseed  = rseed + 1;

    u1_boot = datasample(stream, Uhat_1, size(Uhat_1, 2), 2);
    u2_boot = datasample(stream, Uhat_2, size(Uhat_2, 2), 2);
    u1_boot = u1_boot - mean(u1_boot, 2);
    u2_boot = u2_boot - mean(u2_boot, 2);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test on thetas under homoskedasticity assumption %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Cov_1 = cov(transpose(theta_hat_boot_1));
Cov_2 = cov(transpose(theta_hat_boot_2));
Wald_theta = transpose(theta_hat_1_main - theta_hat_2_main) * ...
             inv(Cov_1 + Cov_2) * (theta_hat_1_main - theta_hat_2_main);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test on betas under heteroskedasticity assumption %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Prepare vectorized bootstrap results
vectorized_B_nG_1 = reshape(B_hat_boot_1, 9, S);
vectorized_B_nG_2 = reshape(B_hat_boot_2, 9, S);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test on betas under heteroskedasticity assumption %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Column 1
[ratio_1_column_1, Cov_standard_1_column_1] = ratios_and_cov_from_boot(B_nG_1, B_hat_boot_1, 1, denom_row);
[ratio_2_column_1, Cov_standard_2_column_1] = ratios_and_cov_from_boot(B_nG_2, B_hat_boot_2, 1, denom_row);

Wald_beta_column_1 = (ratio_1_column_1 - ratio_2_column_1).' * ...
                     ((Cov_standard_1_column_1 + Cov_standard_2_column_1) \ (ratio_1_column_1 - ratio_2_column_1));

% Column 2
[ratio_1_column_2, Cov_standard_1_column_2] = ratios_and_cov_from_boot(B_nG_1, B_hat_boot_1, 2, denom_row);
[ratio_2_column_2, Cov_standard_2_column_2] = ratios_and_cov_from_boot(B_nG_2, B_hat_boot_2, 2, denom_row);

Wald_beta_column_2 = (ratio_1_column_2 - ratio_2_column_2).' * ...
                     ((Cov_standard_1_column_2 + Cov_standard_2_column_2) \ (ratio_1_column_2 - ratio_2_column_2));

% Column 3
[ratio_1_column_3, Cov_standard_1_column_3] = ratios_and_cov_from_boot(B_nG_1, B_hat_boot_1, 3, denom_row);
[ratio_2_column_3, Cov_standard_2_column_3] = ratios_and_cov_from_boot(B_nG_2, B_hat_boot_2, 3, denom_row);

Wald_beta_column_3 = (ratio_1_column_3 - ratio_2_column_3).' * ...
                     ((Cov_standard_1_column_3 + Cov_standard_2_column_3) \ (ratio_1_column_3 - ratio_2_column_3));

%%%%%%%%%%%%%%%%%%
% Output Results %
%%%%%%%%%%%%%%%%%%

fprintf('\n---------------------------------------------\n');
fprintf('Wald Test on Thetas (Homoskedasticity)\n');
fprintf('---------------------------------------------\n');
fprintf('Wald statistic: %.3f\n', Wald_theta);
fprintf('p-value       : %.3f\n\n', 1 - chi2cdf(Wald_theta, 3));

fprintf('---------------------------------------------\n');
fprintf('Wald Tests on Betas (Heteroskedasticity)\n');
fprintf('---------------------------------------------\n');

fprintf('Column 1:\n');
fprintf('  Wald statistic: %.3f\n', Wald_beta_column_1);
fprintf('  p-value       : %.3f\n\n', 1 - chi2cdf(Wald_beta_column_1, 2));

fprintf('Column 2:\n');
fprintf('  Wald statistic: %.3f\n', Wald_beta_column_2);
fprintf('  p-value       : %.3f\n\n', 1 - chi2cdf(Wald_beta_column_2, 2));

fprintf('Column 3:\n');
fprintf('  Wald statistic: %.3f\n', Wald_beta_column_3);
fprintf('  p-value       : %.3f\n\n', 1 - chi2cdf(Wald_beta_column_3, 2));



