% Distance Covariance Test (Bakirov, Rizzo, Szekely 2006)
% In: 
%   X: n x 1 data vector of first strutural shock
%   Y: n x 1 data vector of second strutural shock
%   B: number of permutations (we use 500)
% Out: 
%   Vxy: observed distance covariance
%   pval: permutation-based p-value

function [Vxy, pval] = distance_covariance_test(X, Y, B)

    if size(X,1) ~= size(Y,1)
        error('X and Y must have the same number of observations (rows).');
    end

    n = size(X,1);

    % Compute pairwise Euclidean distance matrices
    DX = squareform(pdist(X));
    DY = squareform(pdist(Y));

    % Double centering of distance matrices
    A = double_center(DX);
    Bmat = double_center(DY);

    % Compute distance covariance
    Vxy = mean(A(:) .* Bmat(:));

    % Permutation test
    perm_stats = zeros(B,1);
    for b = 1:B
        perm = randperm(n);
        DY_perm = DY(perm, perm);
        Bperm = double_center(DY_perm);
        perm_stats(b) = mean(A(:) .* Bperm(:));
    end

    % Compute p-value
    pval = mean(perm_stats >= Vxy);
end

% Helper function: double-center a distance matrix
function C = double_center(D)
    n = size(D,1);
    row_mean = mean(D, 2);
    col_mean = mean(D, 1);
    grand_mean = mean(D(:));
    C = D - row_mean - col_mean + grand_mean;
end
