%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Estimate structural B-Matrix via non-normality    % 
% Based on t-distribution with v degrees of freedom %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

K = size(Uhat, 1);       % Number of variables
v = 4;                   % Degrees of freedom for t-distribution

% Standardize residuals
u_star = chol(SIGMA, 'lower') \ Uhat;  % Dimension: K × T

% Initial guess for theta parameters
theta0 = [-0.25*pi; 0.25*pi; 0.4*pi];  % 3x1 vector for K = 3

% Optimization constraints
A = [];  b = [];        % No inequality constraints
Aeq = []; beq = [];     % No equality constraints
lb = -pi * ones(K,1);   % Lower bounds
ub =  pi * ones(K,1);   % Upper bounds

% Optimization options
options = optimoptions('fmincon', ...
    'Algorithm', 'interior-point', ...   
    'Display', 'off', ...
    'FiniteDifferenceType', 'forward', ...
    'StepTolerance', 1e-10, ...
    'OptimalityTolerance', 1e-8, ...
    'MaxIterations', 1000, ...
    'MaxFunctionEvaluations', 1e5);

% Estimate theta by minimizing negative log-likelihood
theta_hat = fmincon(@(theta) -loglik_t_3x3(u_star, v, theta), ...
                    theta0, A, b, Aeq, beq, lb, ub, [], options);

% Recover structural B-matrix from estimated rotation
B_nG = chol(SIGMA, 'lower') * Qdim3_prime_loglik(theta_hat)';

% Post-estimation: adjust permutation and sign (following Kilian 2009 conventions)
B_nG = B_nG(:, [1, 3, 2]) * diag([-1, -1, 1]);

% chop-off higher decimals to improve reproducibility 
B_nG=chop(B_nG,nod);

% Extract implied Q matrix and rotation angles (theta_hat_main) from B
[~, ~, Q_hat, ~] = rqdecomposition(K, B_nG);     % Orthogonal matrix Q
theta_hat_main = thetas_from_Q_3x3(Q_hat);       % Get back implied thetas

% Display result
disp('B-Matrix from non-normality identification (full sample):');
fmt = [repmat('%10.2f ', 1, size(B_nG,2)) '\n'];
fprintf(fmt, B_nG.');

