%-------------------------------------------------------------
% Estimate structural B-Matrices via non-normality
% for both subsamples
% Based on t-distribution with v degrees of freedom
%-------------------------------------------------------------

K = size(Uhat_1, 1);       % Number of variables
v = 4;                     % Degrees of freedom for t-distribution

% Empirically standardized reduced-form residuals
u_star_1 = chol(SIGMA_1, 'lower') \ Uhat_1;   % Subsample 1
u_star_2 = chol(SIGMA_2, 'lower') \ Uhat_2;   % Subsample 2

% Initial guess for theta rotation angles (K = 3)
theta0 = [-0.25*pi; 0.25*pi; 0.4*pi];

% Optimization constraints
A    = [];  b    = [];
Aeq  = [];  beq  = [];
lb   = -pi * ones(K,1);
ub   =  pi * ones(K,1);

% Optimization options
options = optimoptions('fmincon', ...
    'Algorithm', 'interior-point', ...   
    'Display', 'off', ...
    'FiniteDifferenceType', 'forward', ...
    'StepTolerance', 1e-10, ...
    'OptimalityTolerance', 1e-8, ...
    'MaxIterations', 1000, ...
    'MaxFunctionEvaluations', 1e5);

% Estimate theta for subsample 1 (1973M2–1990M9)
theta_hat_1 = fmincon(@(theta) -loglik_t_3x3(u_star_1, v, theta), ...
                      theta0, A, b, Aeq, beq, lb, ub, [], options);

% Estimate theta for subsample 2 (1990M10–2007M12)
theta_hat_2 = fmincon(@(theta) -loglik_t_3x3(u_star_2, v, theta), ...
                      theta0, A, b, Aeq, beq, lb, ub, [], options);

% Recover structural B-matrices from estimated rotations
B_nG_1 = chol(SIGMA_1, 'lower') * Qdim3_prime_loglik(theta_hat_1)';
B_nG_2 = chol(SIGMA_2, 'lower') * Qdim3_prime_loglik(theta_hat_2)';

% Adjust permutation and sign (based on Kilian conventions)
B_nG_1 = B_nG_1(:, [3, 2, 1]) * diag([1, 1, 1]);
B_nG_2 = B_nG_2(:, [1, 3, 2]) * diag([1, -1, -1]);

% chop-off higher decimals to ensure reproducibility 
B_nG_1=chop(B_nG_1,nod);
B_nG_2=chop(B_nG_2,nod);

% Extract implied Q and rotation angles
[~, ~, Q_hat_1, ~] = rqdecomposition(K, B_nG_1);
[~, ~, Q_hat_2, ~] = rqdecomposition(K, B_nG_2);
theta_hat_1_main = thetas_from_Q_3x3(Q_hat_1);
theta_hat_2_main = thetas_from_Q_3x3(Q_hat_2);

% Display results
disp('-------------------------------------------------------------');
disp('B-Matrix from non-normality identification — Subsample 1:');
fprintf('%10.2f %10.2f %10.2f\n', B_nG_1.');
disp('-------------------------------------------------------------');
disp('B-Matrix from non-normality identification — Subsample 2:');
fprintf('%10.2f %10.2f %10.2f\n', B_nG_2.');
disp('-------------------------------------------------------------');

