%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation function, K = 2, homoskedasticity %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [rej_size10, rej_size5, rej_size1, ...
          rej_power10, rej_power5, rej_power1] = ...
    simulation_2x2_size_power_T_homo(T, L, S, nod)
% H0 (Size):  B2 = B1  (no change in loadings)
% H1 (Power): B2 differs from B1 (change in loadings)

% fmincon options
    options = optimoptions('fmincon', ...
        'Algorithm','interior-point', ...
        'Display','off', ...
        'FiniteDifferenceType','forward', ...
        'OptimalityTolerance',1e-8, ...
        'StepTolerance',1e-10, ...
        'MaxIterations',1000, ...
        'MaxFunctionEvaluations',1e5);

    K = 2;

    % Homoskedastic shock parameters
    mu    = 0;
    sigma = sqrt(64);
    v     = 4;

    % Bounds for theta
    lb = -pi;
    ub =  pi;

    % Wald statistics for size (H0) and power (H1)
    Wald_size  = zeros(1, L);
    Wald_power = zeros(1, L);

    % counter of Monte Carlo replications
    l = 0;

    % global seed for Monte Carlo DGP
    rng(1);

    while l < L

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 1) True parameters under H0 and H1 %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % B1 (true)
        B_aux_1 = [1   -sqrt(3)       -0.5*sqrt(3)   0; ...
                   2    1             -2*sqrt(3)     0; ...
                   1.5 -(3/4)*sqrt(3)  1            0; ...
                   0    0              0            0] * ...
                  [sqrt(64) 0 0 0; ...
                   0 sqrt(2) 0 0; ...
                   0 0 sqrt(2) 0; ...
                   0 0 0 sqrt(2)];

        % B2 under H0: identical to B1
        B_aux_2_H0 = B_aux_1;

        % B2 under H1: changed loadings
        B_aux_2_H1 = [1   -3*sqrt(3)   -0.5*sqrt(3)  0; ...
                      3    1           -2*sqrt(3)    0; ...
                      1.5 -(3/4)*sqrt(3) 1          0; ...
                      0    0            0           0] * ...
                     [sqrt(64) 0 0 0; ...
                      0 sqrt(2) 0 0; ...
                      0 0 sqrt(2) 0; ...
                      0 0 0 sqrt(2)];

        B_1    = B_aux_1(1:K,1:K);
        B_2_H0 = B_aux_2_H0(1:K,1:K);
        B_2_H1 = B_aux_2_H1(1:K,1:K);

        % true theta via rq-decomposition
        [theta_true_1,    ~, ~, ~] = rqdecomposition(K, B_1);
        [theta_true_2_H0, ~, ~, ~] = rqdecomposition(K, B_2_H0);
        [theta_true_2_H1, ~, ~, ~] = rqdecomposition(K, B_2_H1);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 2) Draw shocks and generate reduced-form residuals %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        w_n_1 = random('Normal', mu, sigma, [1, T]);
        w_t_1 = random('T', v, [K-1, T]);
        w_1   = [w_n_1; w_t_1];

        w_n_2 = random('Normal', mu, sigma, [1, T]);
        w_t_2 = random('T', v, [K-1, T]);
        w_2   = [w_n_2; w_t_2];

        % Standardize structural shocks
        w = [diag(sqrt(var(w_1,1,2))) \ (w_1 - mean(w_1,2)), ...
             diag(sqrt(var(w_2,1,2))) \ (w_2 - mean(w_2,2))];

        % Reduced-form shocks
        u1    = B_1    * w(:,1:T);
        u2_H0 = B_2_H0 * w(:,T+1:2*T);
        u2_H1 = B_2_H1 * w(:,T+1:2*T);

        %%%%%%%%%%%%%%%%%%%%%%
        % 3) Main estimation %
        %%%%%%%%%%%%%%%%%%%%%%
        % subsample 1
        u1_boot = u1 - mean(u1,2);
        u_star1 = chol(cov(u1_boot'),'lower') \ u1_boot;

        theta0_1 = theta_true_1;
        A=[]; b=[]; Aeq=[]; beq=[];

        [theta_hat_1_est, ~] = fmincon(@(th) -loglik_t_2x2(u_star1, v, th), ...
                                       theta0_1, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_1 = chol(cov(u1_boot'),'lower') * Qdim_2x2(theta_hat_1_est);
        B_hat_1 = chop(B_hat_1, nod);
        B_hat_1 = ColumnSigns_via_correlation_2x2(B_hat_1, B_1, K);

        if ~isequal(sign(B_hat_1), sign(B_1))
            continue;
        end

        % subsample 2 under H0
        u2_H0_boot = u2_H0 - mean(u2_H0,2);
        u_star2_H0 = chol(cov(u2_H0_boot'),'lower') \ u2_H0_boot;

        theta0_2_H0 = theta_true_2_H0;

        [theta_hat_2_H0_est, ~] = fmincon(@(th) -loglik_t_2x2(u_star2_H0, v, th), ...
                                          theta0_2_H0, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_2_H0 = chol(cov(u2_H0_boot'),'lower') * Qdim_2x2(theta_hat_2_H0_est);
        B_hat_2_H0 = chop(B_hat_2_H0, nod);
        B_hat_2_H0 = ColumnSigns_via_correlation_2x2(B_hat_2_H0, B_2_H0, K);

        if ~isequal(sign(B_hat_2_H0), sign(B_2_H0))
            continue;
        end

        % subsample 2 under H1
        u2_H1_boot = u2_H1 - mean(u2_H1,2);
        u_star2_H1 = chol(cov(u2_H1_boot'),'lower') \ u2_H1_boot;

        theta0_2_H1 = theta_true_2_H1;

        [theta_hat_2_H1_est, ~] = fmincon(@(th) -loglik_t_2x2(u_star2_H1, v, th), ...
                                          theta0_2_H1, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_2_H1 = chol(cov(u2_H1_boot'),'lower') * Qdim_2x2(theta_hat_2_H1_est);
        B_hat_2_H1 = chop(B_hat_2_H1, nod);
        B_hat_2_H1 = ColumnSigns_via_correlation_2x2(B_hat_2_H1, B_2_H1, K);

        if ~isequal(sign(B_hat_2_H1), sign(B_2_H1))
            continue;
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 4) Bootstrap subsample 1 %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%
        [theta1_for_Wald, Cov_1] = bootstrap_theta_second_moment_2x2( ...
            u1, B_1, theta_true_1, v, K, S, options, nod, lb, ub);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 5) Bootstrap subsample 2 under H0 (Size) %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        [theta2_H0_for_Wald, Cov_2_H0] = bootstrap_theta_second_moment_2x2( ...
            u2_H0, B_2_H0, theta_true_2_H0, v, K, S, options, nod, lb, ub);

        dtheta_size   = theta1_for_Wald - theta2_H0_for_Wald;   % scalar
        M_size        = Cov_1 + Cov_2_H0;                       % scalar
        Wald_size(l+1)= (dtheta_size' / M_size) * dtheta_size;  % df=1

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 6) Bootstrap subsample 2 under H1 (Power)
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        [theta2_H1_for_Wald, Cov_2_H1] = bootstrap_theta_second_moment_2x2( ...
            u2_H1, B_2_H1, theta_true_2_H1, v, K, S, options, nod, lb, ub);

        dtheta_power   = theta1_for_Wald - theta2_H1_for_Wald;  % scalar
        M_power        = Cov_1 + Cov_2_H1;                      % scalar
        Wald_power(l+1)= (dtheta_power' / M_power) * dtheta_power;

        l = l + 1;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Empirical rejection rates %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    rej_size10  = mean(Wald_size  > chi2inv(0.90, 1));
    rej_size5   = mean(Wald_size  > chi2inv(0.95, 1));
    rej_size1   = mean(Wald_size  > chi2inv(0.99, 1));

    rej_power10 = mean(Wald_power > chi2inv(0.90, 1));
    rej_power5  = mean(Wald_power > chi2inv(0.95, 1));
    rej_power1  = mean(Wald_power > chi2inv(0.99, 1));
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Helper: bootstrap theta and its variance %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [theta_for_Wald, Cov_theta] = bootstrap_theta_second_moment_2x2( ...
    u, B_ref, theta_true, v, K, S, options, nod, lb, ub)

    theta_boot = zeros(1, S);

    u_boot = u - mean(u,2);

    A=[]; b=[]; Aeq=[]; beq=[];

    s      = 1;
    seed_s = 1;

    while s <= S
        u_star = chol(cov(u_boot'),'lower') \ u_boot;

        theta0 = theta_true;

        [theta_hat_est, ~] = fmincon(@(th) -loglik_t_2x2(u_star, v, th), ...
                                     theta0, A, b, Aeq, beq, lb, ub, [], options);

        B_hat = chol(cov(u_boot'),'lower') * Qdim_2x2(theta_hat_est);
        B_hat = chop(B_hat, nod);
        B_hat = ColumnSigns_via_correlation_2x2(B_hat, B_ref, K);

        if isequal(sign(B_hat), sign(B_ref))
            % retrieve theta
            [theta_hat_est, ~, ~, ~] = rqdecomposition(K, B_hat);
            theta_boot(1, s) = theta_hat_est;
            s = s + 1;
        end

        % Bootstrap residuals
        stream = RandStream('dsfmt19937', 'Seed', seed_s);
        seed_s = seed_s + 1;
        u_boot = datasample(stream, u, size(u,2), 2);
        u_boot = u_boot - mean(u_boot,2);
    end

    Cov_theta      = cov(theta_boot.', 1);  % scalar
    theta_for_Wald = theta_boot(1,1);       % scalar
end
