%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation function, K = 3, heteroskedasticity %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [rej_size10, rej_size5, rej_size1, ...
          rej_power10, rej_power5, rej_power1] = ...
    simulation_3x3_size_power_T_hetero(T, L, S, nod)
% Size:  B2 changes only variances (same loadings as B1)
% Power: B2 changes variances + loadings

% fmincon options
    options = optimoptions('fmincon', ...
        'Algorithm','interior-point', ...
        'Display','off', ...
        'FiniteDifferenceType','forward', ...
        'OptimalityTolerance',1e-8, ...
        'StepTolerance',1e-10, ...
        'MaxIterations',1000, ...
        'MaxFunctionEvaluations',1e5);

    K = 3;

    % seed for Monte Carlo DGP
    seed_l = 1;

    % Wald statistics for size (H0) and power (H1)
    Wald_size  = zeros(1, L);
    Wald_power = zeros(1, L);

    % counter of Monte Carlo replications
    l = 0;
    
    rng(seed_l);
    while l < L

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 1. True parameters (B1, B2 under H0 and H1) %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

        % B1
        B_aux_1 = [1   -sqrt(3)      -0.5*sqrt(3)  0; ...
                   2    1            -2*sqrt(3)    0; ...
                   1.5 -(3/4)*sqrt(3) 1           0; ...
                   0    0             0           0] * ...
                  [sqrt(16) 0 0 0; ...
                   0 sqrt(3) 0 0; ...
                   0 0 sqrt(3) 0; ...
                   0 0 0 sqrt(3)];

        % B2 under H0: same loadings, higher variances
        B_aux_2_H0 = [1   -sqrt(3)      -0.5*sqrt(3)  0; ...
                      2    1            -2*sqrt(3)    0; ...
                      1.5 -(3/4)*sqrt(3) 1           0; ...
                      0    0             0           0] * ...
                     [sqrt(64) 0 0 0; ...
                      0 sqrt(2) 0 0; ...
                      0 0 sqrt(2) 0; ...
                      0 0 0 sqrt(2)];

        % B2 under H1: changed loadings & higher variances
        B_aux_2_H1 = [1   -1.5*sqrt(3)  -0.5*sqrt(3)  0; ...
                      3    1            -2*sqrt(3)    0; ...
                      1.5 -(3/4)*sqrt(3) 1           0; ...
                      0    0             0           0] * ...
                     [sqrt(64) 0 0 0; ...
                      0 sqrt(2) 0 0; ...
                      0 0 sqrt(2) 0; ...
                      0 0 0 sqrt(2)];

        B_1    = B_aux_1(1:K,1:K);
        B_2_H0 = B_aux_2_H0(1:K,1:K);
        B_2_H1 = B_aux_2_H1(1:K,1:K);

        % true thetas via rq-decomposition
        [~, ~, Q_true_1,    ~] = rqdecomposition(K,B_1);
        [~, ~, Q_true_2_H0, ~] = rqdecomposition(K,B_2_H0);
        [~, ~, Q_true_2_H1, ~] = rqdecomposition(K,B_2_H1);

        theta_true_1    = thetas_from_Q_3x3(Q_true_1);
        theta_true_2_H0 = thetas_from_Q_3x3(Q_true_2_H0);
        theta_true_2_H1 = thetas_from_Q_3x3(Q_true_2_H1);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 2. Drawing structural shocks and generating reduced-form residuals %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        mu_1 = 0; sigma_1 = sqrt(16); v_1 = 3;  % regime 1
        mu_2 = 0; sigma_2 = sqrt(64); v_2 = 4;  % regime 2

        % First subsample shocks (regime 1): 1 normal, 2 t
        w_n_1 = random('Normal',mu_1,sigma_1,[1,T]);
        w_t_1 = random('T',v_1,[K-1,T]);
        w_1   = [w_n_1; w_t_1];

        % Second subsample shocks (regime 2)
        w_n_2 = random('Normal',mu_2,sigma_2,[1,T]);
        w_t_2 = random('T',v_2,[K-1,T]);
        w_2   = [w_n_2; w_t_2];

        % Standardize structural shocks
        w = [diag(sqrt(var(w_1,1,2)))\(w_1-mean(w_1,2)) ...
             diag(sqrt(var(w_2,1,2)))\(w_2-mean(w_2,2))];

        % reduced-form shocks
        u1    = B_1    * w(:,1:T);
        u2_H0 = B_2_H0 * w(:,T+1:2*T);
        u2_H1 = B_2_H1 * w(:,T+1:2*T);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 3. Main estimation subsample 1 %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        u1_boot = u1 - mean(u1,2);
        u_star1 = chol(cov(u1_boot'),'lower') \ u1_boot;

        theta0_1 = theta_true_1';  % row vector initial values
        A=[]; b=[]; Aeq=[]; beq=[]; lb=[]; ub=[];

        [theta_hat_1_est, ~] = fmincon(@(theta) -loglik_t_3x3(u_star1,v_1,theta), ...
                                       theta0_1, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_1 = chol(cov(u1_boot'),'lower') * Qdim3_prime_loglik(theta_hat_1_est)';
        B_hat_1 = chop(B_hat_1, nod);
        B_hat_1 = ColumnSigns_via_correlation(B_hat_1,B_1,K);

        if ~isequal(sign(B_hat_1), sign(B_1))
            continue;
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 3b. Main estimation subsample 2 under H0/H1 %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

        % H0
        u2_H0_boot = u2_H0 - mean(u2_H0,2);
        u_star2_H0 = chol(cov(u2_H0_boot'),'lower') \ u2_H0_boot;

        theta0_2_H0 = theta_true_2_H0';

        [theta_hat_2_H0_est, ~] = fmincon(@(theta) -loglik_t_3x3(u_star2_H0,v_2,theta), ...
                                          theta0_2_H0, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_2_H0 = chol(cov(u2_H0_boot'),'lower') * Qdim3_prime_loglik(theta_hat_2_H0_est)';
        B_hat_2_H0 = chop(B_hat_2_H0, nod);
        B_hat_2_H0 = ColumnSigns_via_correlation(B_hat_2_H0,B_2_H0,K);

        if ~isequal(sign(B_hat_2_H0), sign(B_2_H0))
            continue;
        end

        % H1
        u2_H1_boot = u2_H1 - mean(u2_H1,2);
        u_star2_H1 = chol(cov(u2_H1_boot'),'lower') \ u2_H1_boot;

        theta0_2_H1 = theta_true_2_H1';

        [theta_hat_2_H1_est, ~] = fmincon(@(theta) -loglik_t_3x3(u_star2_H1,v_2,theta), ...
                                          theta0_2_H1, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_2_H1 = chol(cov(u2_H1_boot'),'lower') * Qdim3_prime_loglik(theta_hat_2_H1_est)';
        B_hat_2_H1 = chop(B_hat_2_H1, nod);
        B_hat_2_H1 = ColumnSigns_via_correlation(B_hat_2_H1,B_2_H1,K);

        if ~isequal(sign(B_hat_2_H1), sign(B_2_H1))
            continue;
        end

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 4. Bootstrap subsample 1 %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%
        s = 1;
        seed_s = 1;
        B_hat_1_boot = zeros(K,K,S);

        while s <= S
            u_star1 = chol(cov(u1_boot'),'lower') \ u1_boot;

            [theta_hat_1_est, ~] = fmincon(@(theta) -loglik_t_3x3(u_star1,v_1,theta), ...
                                           theta0_1, A, b, Aeq, beq, lb, ub, [], options);

            B_hat_1 = chol(cov(u1_boot'),'lower') * Qdim3_prime_loglik(theta_hat_1_est)';
            B_hat_1 = chop(B_hat_1, nod);
            B_hat_1 = ColumnSigns_via_correlation(B_hat_1,B_1,K);

            if isequal(sign(B_hat_1), sign(B_1))
                B_hat_1_boot(:,:,s) = B_hat_1;
                s = s + 1;
            end

            stream = RandStream('dsfmt19937', 'Seed', seed_s);
            seed_s = seed_s + 1;
            u1_boot = datasample(stream,u1,size(u1,2),2);
            u1_boot = u1_boot - mean(u1_boot,2);
        end

        % Ratios & covariances
        vec_param_1  = reshape(B_hat_1_boot, K^2, S);
        ratios1_boot = [vec_param_1(2,:)./vec_param_1(1,:); ...
                        vec_param_1(3,:)./vec_param_1(1,:)];

        Cov_1 = cov(ratios1_boot.', 1);
        ratio1_for_Wald = ratios1_boot(:,1);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 5. Bootstrap subsample 2 under H0 (Size, H0)  %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        [ratio2_H0_for_Wald, Cov_2_H0] = ...
            bootstrap_second_subsample_hetero_ratio(u2_H0, B_2_H0, theta_true_2_H0, ...
                                                    v_2, K, S, options, nod);

        dtheta_size    = ratio1_for_Wald - ratio2_H0_for_Wald;
        M_size         = Cov_1 + Cov_2_H0;
        Wald_size(l+1) = dtheta_size' * (M_size \ dtheta_size);

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 6. Bootstrap subsample 2 under H1 (Power, H1)  %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        [ratio2_H1_for_Wald, Cov_2_H1] = ...
            bootstrap_second_subsample_hetero_ratio(u2_H1, B_2_H1, theta_true_2_H1, ...
                                                    v_2, K, S, options, nod);

        dtheta_power    = ratio1_for_Wald - ratio2_H1_for_Wald;
        M_power         = Cov_1 + Cov_2_H1;
        Wald_power(l+1) = dtheta_power' * (M_power \ dtheta_power);

        l = l + 1;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Empirical rejection rates %
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    rej_size10  = mean(Wald_size  > chi2inv(0.90, 2));
    rej_size5   = mean(Wald_size  > chi2inv(0.95, 2));
    rej_size1   = mean(Wald_size  > chi2inv(0.99, 2));

    rej_power10 = mean(Wald_power > chi2inv(0.90, 2));
    rej_power5  = mean(Wald_power > chi2inv(0.95, 2));
    rej_power1  = mean(Wald_power > chi2inv(0.99, 2));
end


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Helper: bootstrap second subsample %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [ratio2_for_Wald, Cov_2] = ...
    bootstrap_second_subsample_hetero_ratio(u2, B_2, theta_true_2, ...
                                            v_2, K, S, options, nod)

    B_hat_2_boot = zeros(K,K,S);
    u2_boot = u2 - mean(u2,2);

    A=[]; b=[]; Aeq=[]; beq=[]; lb=[]; ub=[];

    s = 1;
    seed_s = 1;
    while s <= S
        u_star2 = chol(cov(u2_boot'),'lower') \ u2_boot;

        theta0_2 = theta_true_2';

        [theta_hat_2_est, ~] = fmincon(@(theta) -loglik_t_3x3(u_star2,v_2,theta), ...
                                       theta0_2, A, b, Aeq, beq, lb, ub, [], options);

        B_hat_2 = chol(cov(u2_boot'),'lower') * Qdim3_prime_loglik(theta_hat_2_est)';
        B_hat_2 = chop(B_hat_2, nod);
        B_hat_2 = ColumnSigns_via_correlation(B_hat_2,B_2,K);

        if isequal(sign(B_hat_2), sign(B_2))
            B_hat_2_boot(:,:,s) = B_hat_2;
            s = s + 1;
        end

        stream = RandStream('dsfmt19937', 'Seed', seed_s);
        seed_s = seed_s + 1;
        u2_boot = datasample(stream,u2,size(u2,2),2);
        u2_boot = u2_boot - mean(u2_boot,2);
    end

    vec_param_2  = reshape(B_hat_2_boot, K^2, S);
    ratios2_boot = [vec_param_2(2,:)./vec_param_2(1,:); ...
                    vec_param_2(3,:)./vec_param_2(1,:)];

    Cov_2           = cov(ratios2_boot.', 1);
    ratio2_for_Wald = ratios2_boot(:,1);
end